<?php

namespace App\Http\Controllers;

use App\Models\Reservation;
use App\Models\Table;
use Illuminate\Http\Request;

class ReservationController extends Controller
{

public function toggleSeated(\Illuminate\Http\Request $request, \App\Models\Reservation $reservation)
{
    $user = auth()->user();
    if (!$user) abort(401);

    // Hostess + Manager + SuperAdmin can toggle seated
    if (!$user->hasAnyRole(['SuperAdmin', 'Upravitel', 'Hostess'])) {
        abort(403);
    }

    // Optional: if reservation locked_move, still allow seated (only blocks moving tables)
    // If you want seated to also be blocked, tell me and we’ll add that rule.

    $reservation->seated_at = $reservation->seated_at ? null : now();
    $reservation->save();

    return response()->json([
        'ok' => true,
        'seated_at' => $reservation->seated_at,
        'is_seated' => (bool)$reservation->seated_at,
    ]);
}

    public function index(Request $request)
    {
        $date = $request->get('date', now()->toDateString());

        $reservations = Reservation::query()
            ->with(['tables','createdBy','lockedBy'])
            ->whereDate('date', $date)
            ->orderBy('expected_arrival_time')
            ->orderBy('id')
            ->get();

        $tables = Table::query()->orderBy('label')->get();

        return view('reservations.index', compact('reservations','tables','date'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'guest_name' => ['required','string','max:190'],
            'phone' => ['nullable','string','max:50'],
            'people_count' => ['required','integer','min:1','max:50'],
            'date' => ['required','date'],
            'expected_arrival_time' => ['nullable','date_format:H:i'],
            'advance_amount' => ['nullable','numeric','min:0'],
            'notes' => ['nullable','string','max:5000'],
        ]);

        $data['advance_amount'] = $data['advance_amount'] ?? 0;
        $data['created_by_user_id'] = (int)auth()->id();
        $data['status'] = 'pending';

        Reservation::create($data);

        return redirect()->route('reservations.index', ['date' => $data['date']])
            ->with('status', 'Reservation created.');
    }

    public function edit(Reservation $reservation)
    {
        $reservation->load(['tables','createdBy','lockedBy']);
        $tables = Table::query()->orderBy('label')->get();

        return view('reservations.edit', compact('reservation','tables'));
    }

    public function update(Request $request, Reservation $reservation)
    {
        $data = $request->validate([
            'guest_name' => ['required','string','max:190'],
            'phone' => ['nullable','string','max:50'],
            'people_count' => ['required','integer','min:1','max:50'],
            'date' => ['required','date'],
            'expected_arrival_time' => ['nullable','date_format:H:i'],
            'advance_amount' => ['nullable','numeric','min:0'],
            'notes' => ['nullable','string','max:5000'],
            'status' => ['required','in:pending,confirmed,arrived,no_show,cancelled'],
        ]);

        $reservation->update($data);

        return redirect()->route('reservations.index', ['date' => $reservation->date->toDateString()])
            ->with('status', 'Reservation updated.');
    }

    public function updateTables(Request $request, Reservation $reservation)
    {
        $data = $request->validate([
            'table_ids' => ['array'],
            'table_ids.*' => ['integer','exists:tables,id'],
        ]);

        $reservation->tables()->sync($data['table_ids'] ?? []);

        return back()->with('status', 'Tables updated.');
    }

    public function toggleLockMove(Request $request, Reservation $reservation)
    {
        if (!auth()->user()?->hasRole('SuperAdmin') && (int)$reservation->created_by_user_id !== (int)auth()->id()) {
            abort(403);
        }

        $lock = (bool)$request->boolean('locked_move');

        $reservation->locked_move = $lock;
        $reservation->locked_by_user_id = $lock ? (int)auth()->id() : null;
        $reservation->save();

        return back()->with('status', $lock ? 'Move locked.' : 'Move unlocked.');
    }
}
